<?php
/**
 * API endpoint for cleaning unused images - Sử dụng config.php
 */

// Load cấu hình chung
require_once __DIR__ . '/config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    returnJson(['success' => false, 'message' => 'Only POST method allowed'], 405);
    exit();
}

// Xác thực API key từ VPS
$input = json_decode(file_get_contents('php://input'), true);
$apiKey = $input['api_key'] ?? '';

if ($apiKey !== 'muonline2003_api_key_change_this') {
    returnJson(['success' => false, 'message' => 'Unauthorized: Invalid API key'], 401);
    exit();
}

try {
    // Lấy tất cả nội dung tin tức từ database
    $stmt = $pdo->query("SELECT content FROM news");
    $usedImages = [];
    
    while ($row = $stmt->fetch()) {
        $content = $row['content'];
        
        // Tìm tất cả ảnh trong content (các pattern khác nhau)
        $patterns = [
            '/src=["\']([^"\']*\/images\/news\/([^"\'\/]+))["\']/', // src="/public/images/news/filename.jpg"
            '/src=["\']([^"\']*\/uploads\/news\/([^"\'\/]+))["\']/', // src="/uploads/news/filename.jpg" (legacy)
            '/!\[.*?\]\(([^)]*\/images\/news\/([^)\/]+))\)/', // Markdown ![alt](url)
            '/!\[.*?\]\(([^)]*\/uploads\/news\/([^)\/]+))\)/' // Markdown legacy
        ];
        
        foreach ($patterns as $pattern) {
            preg_match_all($pattern, $content, $matches);
            
            if (!empty($matches[2])) {
                foreach ($matches[2] as $filename) {
                    $usedImages[$filename] = true;
                }
            }
        }
    }
    
    // Lấy tất cả ảnh trong thư mục uploads
    $uploadDir = '../public/images/news/';
    $deletedImages = [];
    $totalDeleted = 0;
    $totalSize = 0;
    
    // Debug: Kiểm tra xem đang chạy trên hosting hay local
    $isHosting = (strpos($_SERVER['HTTP_HOST'] ?? '', 'museson2.com') !== false);
    
    if ($isHosting) {
        // Trên hosting thực tế - cần xác định đúng đường dẫn
        $uploadDir = '../public/images/news/';
    } else {
        // Local development - có thể khác
        $uploadDir = '../public/images/news/';
    }
    
    // Tạo thư mục nếu chưa tồn tại
    if (!is_dir($uploadDir)) {
        if (!mkdir($uploadDir, 0755, true)) {
            returnJson([
                'success' => false,
                'message' => 'Không thể tạo thư mục news: ' . $uploadDir
            ], 500);
            exit();
        }
    }
    
    // Kiểm tra quyền ghi của thư mục
    if (!is_writable($uploadDir)) {
        returnJson([
            'success' => false,
            'message' => 'Thư mục không có quyền ghi: ' . $uploadDir . '. Kiểm tra chmod 755'
        ], 500);
        exit();
    }
    
    if (is_dir($uploadDir)) {
        $files = scandir($uploadDir);
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..' || $file === '.gitkeep') {
                continue;
            }
            
            $filePath = $uploadDir . $file;
            if (is_file($filePath)) {
                // Check if it's an image file
                $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'jfif'];
                $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                
                if (in_array($extension, $imageExtensions)) {
                    // Nếu ảnh không được sử dụng trong tin tức nào
                    if (!isset($usedImages[$file])) {
                        $fileSize = filesize($filePath);
                        
                        // Kiểm tra quyền xóa file
                        if (!is_writable($filePath)) {
                            error_log("Cannot delete file (no permission): {$filePath}");
                            continue;
                        }
                        
                        if (unlink($filePath)) {
                            $deletedImages[] = [
                                'filename' => $file,
                                'size' => $fileSize,
                                'size_formatted' => formatFileSize($fileSize)
                            ];
                            $totalDeleted++;
                            $totalSize += $fileSize;
                        } else {
                            error_log("Failed to delete file: {$filePath}");
                        }
                    }
                }
            }
        }
    }
    
    returnJson([
        'success' => true,
        'message' => "Dọn dẹp hoàn tất! Đã xóa {$totalDeleted} ảnh không sử dụng.",
        'deleted_count' => $totalDeleted,
        'total_size_freed' => formatFileSize($totalSize),
        'deleted_images' => $deletedImages,
        'used_images_count' => count($usedImages),
        'debug' => [
            'upload_dir' => $uploadDir,
            'is_hosting' => $isHosting ?? false,
            'dir_exists' => is_dir($uploadDir),
            'dir_writable' => is_writable($uploadDir),
            'total_files_scanned' => count(scandir($uploadDir)) - 2 // trừ . và ..
        ]
    ]);
    
} catch (Exception $e) {
    returnJson([
        'success' => false,
        'message' => 'Lỗi khi dọn dẹp: ' . $e->getMessage()
    ], 500);
}

function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}
?>
