<?php
// Load cấu hình chung
require_once 'config.php';

// Helper function - sử dụng returnError từ config.php
function returnError($message, $statusCode = 400) {
    returnJson(['success' => false, 'message' => $message], $statusCode);
}

// Lấy method HTTP
$method = $_SERVER['REQUEST_METHOD'];

// Xử lý theo method
switch($method) {
    case 'GET':
        handleGet();
        break;
    case 'POST':
        handlePost();
        break;
    case 'PUT':
        handlePut();
        break;
    case 'DELETE':
        handleDelete();
        break;
    default:
        returnError('Method không được hỗ trợ', 405);
}

// Xử lý GET - Lấy danh sách tin tức
function handleGet() {
    global $pdo;
    
    try {
        // Lấy tham số từ URL
        $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
        $category = isset($_GET['category']) ? $_GET['category'] : 'all';
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 10;
        $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
        $status = isset($_GET['status']) ? $_GET['status'] : 'published'; // Mặc định chỉ lấy tin published
        $search = isset($_GET['search']) ? trim($_GET['search']) : '';
        $incrementView = isset($_GET['increment_view']) ? $_GET['increment_view'] === 'true' : true; // Mặc định tăng view
        
        // Hỗ trợ legacy action parameter từ HostingApiClient
        $action = isset($_GET['action']) ? $_GET['action'] : '';
        
        if ($action === 'get' || $id) {
            // Lấy tin tức theo ID
            $stmt = $pdo->prepare("SELECT * FROM news WHERE id = ?");
            $stmt->execute([$id]);
            $news = $stmt->fetch();
            
            if ($news) {
                // Chỉ tăng lượt xem khi increment_view = true và tin tức đã published
                if ($incrementView && $news['status'] === 'published') {
                    // Simple rate limiting - tránh spam views
                    $clientIp = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'unknown';
                    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
                    $cacheKey = "news_view_" . $id . "_" . md5($clientIp . $userAgent);
                    $cacheFile = sys_get_temp_dir() . '/' . $cacheKey . '.tmp';
                    
                    $shouldIncrement = true;
                    if (file_exists($cacheFile) && (time() - filemtime($cacheFile)) < 300) {
                        $shouldIncrement = false; // Đã xem trong 5 phút
                    }
                    
                    if ($shouldIncrement) {
                        $updateViewStmt = $pdo->prepare("UPDATE news SET views = views + 1 WHERE id = ?");
                        $updateViewStmt->execute([$id]);
                        
                        // Lưu cache
                        file_put_contents($cacheFile, time());
                        
                        // Lấy lại dữ liệu sau khi cập nhật views
                        $stmt = $pdo->prepare("SELECT * FROM news WHERE id = ?");
                        $stmt->execute([$id]);
                        $news = $stmt->fetch();
                    }
                }
                
                returnJson([
                    'success' => true, 
                    'data' => $news,
                    'view_incremented' => $incrementView && $news['status'] === 'published'
                ]);
            } else {
                returnError('Không tìm thấy tin tức', 404);
            }
        } else {
            // Lấy danh sách tin tức với phân trang
            // Hỗ trợ legacy action=list từ HostingApiClient
            $whereConditions = [];
            $params = [];
            
            // Filter theo status (luôn có vì đã set default = 'published')
            // Nếu status = 'all' thì không filter (cho admin)
            if ($status !== 'all') {
                $whereConditions[] = "status = ?";
                $params[] = $status;
            }
            
            // Filter theo category nếu có và không phải 'all'
            if ($category !== '' && $category !== 'all') {
                $whereConditions[] = "category = ?";
                $params[] = $category;
            }
            
            // Search theo title hoặc content nếu có
            if ($search !== '') {
                $whereConditions[] = "(title LIKE ? OR content LIKE ?)";
                $params[] = '%' . $search . '%';
                $params[] = '%' . $search . '%';
            }
            
            // Build SQL với WHERE conditions
            $whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
            $sql = "SELECT * FROM news {$whereClause} ORDER BY priority ASC, created_at DESC LIMIT $limit OFFSET $offset";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $newsList = $stmt->fetchAll();
            
            // Transform data to match React component format
            $transformedNews = array_map(function($news) {
                // Get category from database, default to 'announcement' if not set
                $category = $news['category'] ?? 'announcement';
                
                // Map category to label
                $labelMap = [
                    'announcement' => 'THÔNG BÁO',
                    'event' => 'SỰ KIỆN', 
                    'update' => 'CẬP NHẬT',
                    'guide' => 'HƯỚNG DẪN',
                    'maintenance' => 'BẢO TRÌ'
                ];
                
                $label = $labelMap[$category] ?? 'THÔNG BÁO';
                
                return [
                    'id' => $news['id'],
                    'title' => $news['title'],
                    'content' => $news['content'],
                    'image_url' => $news['image_url'] ?? '',
                    'author' => $news['author'] ?? 'Admin',
                    'status' => $news['status'] ?? 'published',
                    'created_at' => $news['created_at'],
                    'updated_at' => $news['updated_at'] ?? $news['created_at'],
                    // Add fields required by React component (using defaults)
                    'category' => $category,
                    'label' => $label,
                    'date' => date('d/m/Y', strtotime($news['created_at'])),
                    'summary' => mb_substr(strip_tags($news['content']), 0, 150) . '...',
                    'priority' => (int)($news['priority'] ?? 5), // Lấy priority từ database, default = 5
                    'views' => (int)$news['views'], // Lấy views từ database
                    'featured' => 0 // Default not featured
                ];
            }, $newsList);
            
            // Filter by category if not 'all'
            if ($category !== 'all') {
                $transformedNews = array_filter($transformedNews, function($news) use ($category) {
                    return $news['category'] === $category;
                });
                // Re-index array after filtering
                $transformedNews = array_values($transformedNews);
            }
            
            // Đếm tổng số tin tức với filters
            $countWhereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
            $countSql = "SELECT COUNT(*) as total FROM news {$countWhereClause}";
            $countStmt = $pdo->prepare($countSql);
            $countStmt->execute($params);
            $total = $countStmt->fetch()['total'];
            
            returnJson([
                'success' => true,
                'data' => $transformedNews,
                'pagination' => [
                    'total' => (int)$total,
                    'limit' => $limit,
                    'offset' => $offset,
                    'has_more' => ($offset + $limit) < $total
                ]
            ]);
        }
    } catch(PDOException $e) {
        returnError('Lỗi truy vấn cơ sở dữ liệu: ' . $e->getMessage(), 500);
    }
}

// Xử lý POST - Tạo tin tức mới
function handlePost() {
    global $pdo;
    
    try {
        // Lấy dữ liệu JSON từ request body
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            returnError('Dữ liệu JSON không hợp lệ');
        }
        
        // Validate dữ liệu bắt buộc
        if (empty($input['title']) || empty($input['content'])) {
            returnError('Tiêu đề và nội dung là bắt buộc');
        }
        
        // Chuẩn bị dữ liệu
        $title = trim($input['title']);
        $content = trim($input['content']);
        $image_url = isset($input['image_url']) ? trim($input['image_url']) : '';
        $author = isset($input['author']) ? trim($input['author']) : 'Admin';
        $status = isset($input['status']) ? $input['status'] : 'published';
        $views = isset($input['views']) ? (int)$input['views'] : 0; // Thêm views
        $priority = isset($input['priority']) ? (int)$input['priority'] : 5; // Thêm priority
        $category = isset($input['category']) ? $input['category'] : 'announcement'; // Thêm category
        
        // Thêm tin tức vào database
        $stmt = $pdo->prepare("
            INSERT INTO news (title, content, image_url, author, status, views, priority, category, created_at, updated_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
        ");
        
        $result = $stmt->execute([$title, $content, $image_url, $author, $status, $views, $priority, $category]);
        
        if ($result) {
            $newId = $pdo->lastInsertId();
            
            // Lấy tin tức vừa tạo để trả về
            $stmt = $pdo->prepare("SELECT * FROM news WHERE id = ?");
            $stmt->execute([$newId]);
            $newNews = $stmt->fetch();
            
            returnJson([
                'success' => true,
                'message' => 'Tạo tin tức thành công',
                'data' => $newNews
            ], 201);
        } else {
            returnError('Không thể tạo tin tức', 500);
        }
    } catch(PDOException $e) {
        returnError('Lỗi cơ sở dữ liệu: ' . $e->getMessage(), 500);
    }
}

// Xử lý PUT - Cập nhật tin tức
function handlePut() {
    global $pdo;
    
    try {
        $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
        
        if (!$id) {
            returnError('ID tin tức là bắt buộc');
        }
        
        // Lấy dữ liệu JSON
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            returnError('Dữ liệu JSON không hợp lệ');
        }
        
        // Kiểm tra tin tức có tồn tại không
        $stmt = $pdo->prepare("SELECT id FROM news WHERE id = ?");
        $stmt->execute([$id]);
        
        if (!$stmt->fetch()) {
            returnError('Không tìm thấy tin tức', 404);
        }
        
        // Chuẩn bị dữ liệu cập nhật
        $updateFields = [];
        $updateValues = [];
        
        if (isset($input['title'])) {
            $updateFields[] = "title = ?";
            $updateValues[] = trim($input['title']);
        }
        
        if (isset($input['content'])) {
            $updateFields[] = "content = ?";
            $updateValues[] = trim($input['content']);
        }
        
        if (isset($input['image_url'])) {
            $updateFields[] = "image_url = ?";
            $updateValues[] = trim($input['image_url']);
        }
        
        if (isset($input['author'])) {
            $updateFields[] = "author = ?";
            $updateValues[] = trim($input['author']);
        }
        
        if (isset($input['status'])) {
            $updateFields[] = "status = ?";
            $updateValues[] = $input['status'];
        }
        
        if (isset($input['views'])) {
            $updateFields[] = "views = ?";
            $updateValues[] = (int)$input['views'];
        }
        
        if (isset($input['priority'])) {
            $updateFields[] = "priority = ?";
            $updateValues[] = (int)$input['priority'];
        }
        
        if (isset($input['category'])) {
            $updateFields[] = "category = ?";
            $updateValues[] = $input['category'];
        }
        // nên sẽ bỏ qua các trường này
        
        if (empty($updateFields)) {
            returnError('Không có dữ liệu để cập nhật');
        }
        
        // Thêm updated_at
        $updateFields[] = "updated_at = NOW()";
        $updateValues[] = $id;
        
        // Thực hiện cập nhật
        $sql = "UPDATE news SET " . implode(', ', $updateFields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute($updateValues);
        
        if ($result) {
            // Lấy tin tức đã cập nhật
            $stmt = $pdo->prepare("SELECT * FROM news WHERE id = ?");
            $stmt->execute([$id]);
            $updatedNews = $stmt->fetch();
            
            returnJson([
                'success' => true,
                'message' => 'Cập nhật tin tức thành công',
                'data' => $updatedNews
            ]);
        } else {
            returnError('Không thể cập nhật tin tức', 500);
        }
    } catch(PDOException $e) {
        returnError('Lỗi cơ sở dữ liệu: ' . $e->getMessage(), 500);
    }
}

// Xử lý DELETE - Xóa tin tức
function handleDelete() {
    global $pdo;
    
    try {
        // Lấy ID từ URL hoặc request body
        $id = null;
        
        // Kiểm tra từ URL parameter trước
        if (isset($_GET['id'])) {
            $id = (int)$_GET['id'];
        } else {
            // Nếu không có trong URL, kiểm tra request body
            $input = json_decode(file_get_contents('php://input'), true);
            if (isset($input['id'])) {
                $id = (int)$input['id'];
            }
        }
        
        if (!$id) {
            returnError('ID tin tức là bắt buộc');
        }
        
        // Kiểm tra tin tức có tồn tại không
        $stmt = $pdo->prepare("SELECT * FROM news WHERE id = ?");
        $stmt->execute([$id]);
        $news = $stmt->fetch();
        
        if (!$news) {
            returnError('Không tìm thấy tin tức', 404);
        }
        
        // Xóa tin tức
        $stmt = $pdo->prepare("DELETE FROM news WHERE id = ?");
        $result = $stmt->execute([$id]);
        
        if ($result) {
            returnJson([
                'success' => true,
                'message' => 'Xóa tin tức thành công',
                'deleted_news' => $news
            ]);
        } else {
            returnError('Không thể xóa tin tức', 500);
        }
    } catch(PDOException $e) {
        returnError('Lỗi cơ sở dữ liệu: ' . $e->getMessage(), 500);
    }
}
?>
